"""
AI Image Analysis Service using g4f with PollinationsAI
Provides real image analysis for medical documents and skin conditions
"""
from g4f.client import Client
from g4f.Provider import PollinationsAI

# Initialize g4f client with PollinationsAI (no API key required)
client = Client(provider=PollinationsAI)


def analyze_medical_image(image_bytes: bytes, filename: str, analysis_type: str = 'general', context: str = '', language: str = 'en') -> dict:
    """
    Analyze a medical image using g4f PollinationsAI
    
    Args:
        image_bytes: The image file bytes
        filename: Original filename
        analysis_type: Type of analysis (skin, document, xray, etc.)
        context: Additional context about the image
        language: Response language ('en' for English, 'ne' for Nepali, etc.)
    
    Returns:
        dict with analysis results
    """
    
    # Language instruction
    language_names = {'en': 'English', 'ne': 'Nepali (नेपाली)', 'hi': 'Hindi (हिन्दी)'}
    lang_name = language_names.get(language, 'English')
    lang_instruction = f"\n\nIMPORTANT: Respond ENTIRELY in {lang_name} language." if language != 'en' else ""
    
    # Build analysis prompt based on type
    prompts = {
        'skin': f"""You are an expert dermatologist AI assistant. Analyze this skin image and provide:

1. **Observations**: Describe what you see in detail (color, texture, patterns, location)
2. **Possible Conditions**: List 2-3 most likely conditions based on visual analysis
3. **Severity Assessment**: Rate as mild/moderate/severe
4. **Immediate Care**: Simple home care recommendations
5. **When to See a Doctor**: Warning signs that need professional attention

Additional context from user: {context if context else 'None provided'}

IMPORTANT: Always include a disclaimer that this is AI-generated guidance and not a medical diagnosis. Recommend consulting a dermatologist for accurate diagnosis.{lang_instruction}""",

        'prescription': f"""You are a prescription analysis AI. Analyze this prescription image and provide:

1. **Medications Listed**: List all medicines with their dosages
2. **Usage Instructions**: Frequency and timing for each medicine
3. **Duration**: How long to take each medication
4. **Precautions**: Important warnings or side effects to watch for
5. **General Notes**: Any other instructions on the prescription

Additional context: {context if context else 'None provided'}

Include a disclaimer about verifying with your pharmacist.{lang_instruction}""",

        'document': f"""You are a medical document analysis AI. Analyze this medical document image and provide:

1. **Document Type**: Identify what type of document this is (lab report, prescription, discharge summary, etc.)
2. **Key Information**: Extract important details like dates, patient info, doctor info
3. **Medical Values**: List any test results, measurements, or values found
4. **Summary**: Brief summary of the document's purpose and findings
5. **Notable Points**: Any abnormal values or important notes

Additional context: {context if context else 'None provided'}

Format your response clearly with headers. Include a disclaimer about AI analysis limitations.{lang_instruction}""",

        'xray': f"""You are a radiology AI assistant. Analyze this X-ray image and provide:

1. **Image Quality**: Assess the quality and positioning of the X-ray
2. **Anatomical Observations**: Describe bone structures, soft tissues visible
3. **Findings**: Note any abnormalities or areas of concern
4. **Impression**: Overall assessment based on observations
5. **Recommendations**: Suggest follow-up if needed

Additional context: {context if context else 'None provided'}

DISCLAIMER: AI-assisted preliminary analysis. Always requires radiologist interpretation for diagnosis.{lang_instruction}""",

        'lab_report': f"""You are a medical lab report analyst AI. Analyze this lab report image and provide:

1. **Report Type**: Type of lab tests (CBC, metabolic panel, lipid profile, etc.)
2. **Test Results**: Extract all test values with their reference ranges
3. **Normal/Abnormal**: Clearly indicate which values are within/outside normal range
4. **Clinical Significance**: Explain what abnormal values might indicate
5. **Recommendations**: General health recommendations based on results

Additional context: {context if context else 'None provided'}

Note: AI analysis for informational purposes. Consult your physician for interpretation.{lang_instruction}""",

        'ecg': f"""You are a cardiology AI assistant. Analyze this ECG/EKG image and provide:

1. **Rhythm**: Identify the heart rhythm (regular/irregular, sinus/other)
2. **Rate**: Estimate heart rate if visible
3. **Waveform Analysis**: Describe P waves, QRS complex, T waves
4. **Findings**: Note any abnormalities or patterns
5. **Recommendations**: Suggested follow-up if needed

Additional context: {context if context else 'None provided'}

DISCLAIMER: This is AI-assisted preliminary analysis only. A cardiologist must interpret ECGs for diagnosis.{lang_instruction}""",

        'general': f"""You are a medical image analysis AI assistant. Analyze this medical image and provide:

1. **Image Description**: What type of medical image is this?
2. **Key Observations**: What can you see in this image?
3. **Analysis**: Provide relevant medical insights
4. **Recommendations**: Any suggested next steps
5. **Important Notes**: Things the user should be aware of

Additional context from user: {context if context else 'None provided'}

Include appropriate medical disclaimers in your response.{lang_instruction}"""
    }
    
    prompt = prompts.get(analysis_type, prompts['general'])
    
    try:
        images = [[image_bytes, filename]]
        
        response = client.chat.completions.create(
            model="openai",
            messages=[
                {"role": "user", "content": prompt}
            ],
            images=images,
        )
        
        analysis_text = response.choices[0].message.content
        
        return {
            'success': True,
            'analysis': analysis_text,
            'analysis_type': analysis_type,
            'model': 'g4f-pollinations-ai'
        }
        
    except Exception as e:
        print(f"[AI Image Analysis Error] {str(e)}")
        return {
            'success': False,
            'error': str(e),
            'analysis': get_fallback_analysis(analysis_type),
            'analysis_type': analysis_type,
            'model': 'fallback'
        }


def get_fallback_analysis(analysis_type: str) -> str:
    """Return fallback analysis text when AI service is unavailable"""
    
    fallbacks = {
        'skin': """**AI Image Analysis - Service Temporarily Unavailable**

We were unable to analyze your image at this time. Please try again later.

**General Skin Care Recommendations:**
• Keep the affected area clean and dry
• Avoid scratching or irritating the area
• Watch for signs of infection (increasing redness, warmth, pus)
• If symptoms persist or worsen, consult a dermatologist

*For accurate diagnosis, please visit a healthcare provider.*""",

        'document': """**Medical Document Uploaded**

Your document has been saved to your medical records. AI analysis is temporarily unavailable.

The document will be available for:
• Your personal reference
• Sharing with healthcare providers
• AI analysis when the service is restored

*Please consult your doctor for interpretation of medical documents.*""",

        'xray': """**X-Ray Image Received**

AI analysis is temporarily unavailable. Your image has been saved for your records.

**General Information:**
• X-ray interpretation requires professional radiologist review
• Keep this image for your medical records
• Share with your healthcare provider for proper diagnosis

*This image requires professional medical interpretation.*""",

        'general': """**Medical Image Analysis Unavailable**

We were unable to analyze your image at this time. Please try again later or consult with a healthcare provider for proper assessment.

Your image has been saved to your medical records for future reference.

*For medical concerns, please consult a qualified healthcare professional.*"""
    }
    
    return fallbacks.get(analysis_type, fallbacks['general'])


def analyze_document_for_storage(image_bytes: bytes, filename: str, document_type: str) -> dict:
    """
    Analyze a medical document and extract key information for storage
    
    Returns structured data for database storage
    """
    
    # Map document types to analysis types
    doc_type_mapping = {
        'lab_report': 'lab_report',
        'blood_test': 'lab_report',
        'xray': 'xray',
        'mri': 'xray',
        'ct_scan': 'xray',
        'ultrasound': 'xray',
        'ecg': 'document',
        'prescription': 'document',
        'discharge_summary': 'document',
        'pathology': 'lab_report',
    }
    
    analysis_type = doc_type_mapping.get(document_type, 'document')
    
    result = analyze_medical_image(
        image_bytes=image_bytes,
        filename=filename,
        analysis_type=analysis_type,
        context=f"Document type: {document_type}"
    )
    
    analysis_text = result.get('analysis', '')
    
    # Generate a short summary (first 200 chars of significant content)
    lines = [l.strip() for l in analysis_text.split('\n') if l.strip() and not l.startswith('*')]
    summary = ' '.join(lines)[:200] + '...' if len(' '.join(lines)) > 200 else ' '.join(lines)
    
    return {
        'ai_analysis': analysis_text,
        'ai_summary': summary,
        'ocr_text': None,  # Could add OCR integration later
        'analysis_success': result.get('success', False)
    }
