"""
Live AI Call Routes for Swasthya Healthcare
Provides voice-based AI medical consultations with:
- Multi-language support (English, Hindi, Nepali)
- Healthcare-focused AI responses
- Text-to-speech for AI responses
- Session management for ongoing consultations
"""

from flask import Blueprint, request, jsonify, g, send_file
from flask_jwt_extended import jwt_required, get_jwt_identity
import os
import uuid
import asyncio
import logging
from datetime import datetime, timedelta
from typing import Dict, List, Optional
from dataclasses import dataclass, field
from collections import deque
import hashlib
import re

import g4f
from g4f.client import AsyncClient

from app import db

logger = logging.getLogger(__name__)

# Create blueprint
live_ai_bp = Blueprint('live_ai', __name__)

# Import TTS service with Edge TTS
try:
    from app.utils.tts_service import (
        get_speech_generator, generate_speech_async,
        get_optimal_voice, AUDIO_OUTPUT_DIR, LANGUAGE_NAMES, clean_text_for_tts,
        EDGE_TTS_AVAILABLE
    )
    TTS_AVAILABLE = EDGE_TTS_AVAILABLE
    speech_generator = get_speech_generator()
except ImportError as e:
    print(f"Warning: TTS service import failed: {e}")
    TTS_AVAILABLE = False
    speech_generator = None
    AUDIO_OUTPUT_DIR = "/tmp/swasthya_audio"
    LANGUAGE_NAMES = {'en-US': 'English', 'hi-IN': 'Hindi', 'ne-NP': 'Nepali'}
    os.makedirs(AUDIO_OUTPUT_DIR, exist_ok=True)

# Healthcare AI Specialists configuration
MEDICAL_SPECIALISTS = {
    'physician': {
        'name': 'Dr. Sathi',
        'specialty': 'General Medicine',
        'prompt': """You are Dr. Sathi, a compassionate virtual general physician speaking to patients.
        
Your approach:
1. Listen carefully to symptoms and concerns
2. Ask clarifying questions about duration, severity, and associated symptoms
3. Provide clear, reassuring explanations in simple language
4. Give practical advice for symptom management
5. Always recommend professional consultation for serious concerns

Keep responses concise (2-3 sentences) for voice conversation.
Always remind patients this is preliminary guidance, not diagnosis."""
    },
    'psychiatrist': {
        'name': 'Dr. Mind Care',
        'specialty': 'Mental Health',
        'prompt': """You are Dr. Mind Care, a compassionate mental health support specialist.
        
Your approach:
1. Create a safe, non-judgmental space for sharing feelings
2. Practice active listening and validate emotions
3. Offer coping strategies and mindfulness techniques
4. Recognize when professional help is needed
5. Provide crisis resources when appropriate

Keep responses warm and supportive (2-3 sentences) for voice conversation.
Always emphasize that talking to a professional therapist is beneficial."""
    },
    'dermatologist': {
        'name': 'Dr. Skin Expert',
        'specialty': 'Dermatology',
        'prompt': """You are Dr. Skin Expert, a knowledgeable virtual dermatologist.
        
Your approach:
1. Ask about symptom location, appearance, duration, and triggers
2. Provide general guidance on skin care
3. Recommend when in-person dermatologist visit is needed
4. Suggest over-the-counter remedies when appropriate

Keep responses concise (2-3 sentences) for voice conversation.
Always recommend professional examination for persistent or concerning conditions."""
    },
    'pediatrician': {
        'name': 'Dr. Child Care',
        'specialty': 'Pediatrics',
        'prompt': """You are Dr. Child Care, a warm and reassuring virtual pediatrician.
        
Your approach:
1. Always ask for child's age first - it affects all advice
2. Be sensitive to parental concerns
3. Provide age-appropriate guidance
4. Highlight red flags requiring immediate care
5. Reassure parents while being thorough

Keep responses warm and clear (2-3 sentences) for voice conversation.
Trust parental instinct and always err on the side of caution with children."""
    },
    'nutritionist': {
        'name': 'Dr. Nutrition',
        'specialty': 'Nutrition & Diet',
        'prompt': """You are Dr. Nutrition, a certified nutritionist focused on healthy eating.
        
Your approach:
1. Understand current eating patterns and goals
2. Consider medical conditions and restrictions
3. Provide practical, sustainable dietary advice
4. Focus on whole foods and balanced nutrition
5. Avoid promoting fad diets

Keep responses practical (2-3 sentences) for voice conversation.
Recommend registered dietitian consultation for complex medical nutrition needs."""
    },
    'cardiologist': {
        'name': 'Dr. Heart Care',
        'specialty': 'Cardiology',
        'prompt': """You are Dr. Heart Care, a knowledgeable virtual cardiologist.
        
Your approach:
1. Take ALL heart-related symptoms seriously
2. Recognize emergency symptoms requiring immediate care
3. Discuss risk factors and prevention
4. Provide heart-healthy lifestyle guidance
5. Encourage regular check-ups

Keep responses clear and calm (2-3 sentences) for voice conversation.
IMPORTANT: Advise calling emergency services for chest pain, severe breathlessness, or fainting."""
    }
}

MEDICAL_DISCLAIMER = "⚠️ This is AI-generated health guidance, not medical diagnosis. Please consult a licensed healthcare provider for proper medical advice."


@dataclass
class LiveCallSession:
    """Session for live AI medical consultations"""
    session_id: str
    user_id: Optional[int]
    specialist: str
    created_at: datetime
    last_activity: datetime
    conversation_history: deque = field(default_factory=lambda: deque(maxlen=50))
    language_code: str = "en-US"
    total_interactions: int = 0
    patient_context: str = ""
    
    def to_dict(self):
        return {
            'session_id': self.session_id,
            'specialist': self.specialist,
            'specialist_name': MEDICAL_SPECIALISTS.get(self.specialist, {}).get('name', 'Dr. AI'),
            'created_at': self.created_at.isoformat(),
            'total_interactions': self.total_interactions,
            'language': self.language_code
        }
    
    def get_history(self, limit: int = 10) -> List[Dict]:
        """Get recent conversation history"""
        return list(self.conversation_history)[-limit:]


# Active sessions storage (in production, use Redis or database)
active_sessions: Dict[str, LiveCallSession] = {}


def cleanup_old_sessions():
    """Remove sessions older than 1 hour"""
    cutoff = datetime.now() - timedelta(hours=1)
    expired = [sid for sid, s in active_sessions.items() if s.last_activity < cutoff]
    for sid in expired:
        del active_sessions[sid]


def generate_medical_response(session: LiveCallSession, user_input: str) -> str:
    """Generate AI response for medical consultation"""
    
    specialist_config = MEDICAL_SPECIALISTS.get(session.specialist, MEDICAL_SPECIALISTS['physician'])
    system_prompt = specialist_config['prompt']
    
    logger.info(f"Generating response for specialist: {session.specialist} ({specialist_config.get('name', 'Dr. AI')})")
    
    # Get language instruction
    language_name = LANGUAGE_NAMES.get(session.language_code, 'English')
    language_instruction = ""
    if session.language_code != 'en-US':
        language_instruction = f"\n\nIMPORTANT: Respond ENTIRELY in {language_name}. Do not mix languages."
    
    # Include patient context from previous text chat
    patient_context_section = ""
    if session.patient_context:
        patient_context_section = f"\nContext from patient's previous conversation:\n{session.patient_context}\n"
    
    # Build conversation context
    history_context = ""
    if session.conversation_history:
        history_parts = []
        for msg in list(session.conversation_history)[-5:]:
            role = "Patient" if msg.get('role') == 'user' else "You"
            history_parts.append(f"{role}: {msg.get('content', '')}")
        history_context = "\n".join(history_parts)
    
    # Create prompt
    prompt = f"""{system_prompt}
{language_instruction}
{patient_context_section}
{f"Previous conversation:\\n{history_context}\\n" if history_context else ""}
Patient's Current Concern: "{user_input}"

Respond naturally and helpfully. Keep your response concise (2-3 sentences) for voice conversation.
End with a follow-up question or suggestion when appropriate."""

    try:
        # Use g4f for AI response (sync version)
        response = g4f.ChatCompletion.create(
            model='deepseek-r1',
            messages=[{"role": "user", "content": prompt}],
        )
        
        # g4f returns string directly
        if isinstance(response, str):
            ai_response = response.strip()
        else:
            ai_response = response.choices[0].message.content.strip()
        
        # Clean for voice output
        ai_response = clean_for_voice(ai_response)
        
        return ai_response
        
    except Exception as e:
        logger.error(f"Error generating medical response: {e}")
        return "I apologize, but I'm having trouble connecting right now. Please try again in a moment, or if you have urgent concerns, please contact a healthcare provider directly."


def clean_for_voice(text: str) -> str:
    """Clean response for natural voice output"""
    # Strip <think>...</think> tags from DeepSeek R1 responses
    text = re.sub(r'<think>[\s\S]*?</think>', '', text, flags=re.IGNORECASE)
    
    # Remove markdown
    text = re.sub(r'\*\*([^*]+)\*\*', r'\1', text)
    text = re.sub(r'\*([^*]+)\*', r'\1', text)
    text = re.sub(r'^- ', '', text, flags=re.MULTILINE)
    text = re.sub(r'^\d+\. ', '', text, flags=re.MULTILINE)
    
    # Improve natural speech
    text = text.replace('...', ', ')
    text = text.replace('  ', ' ')
    
    return text.strip()


@live_ai_bp.route('/live-call/start', methods=['POST'])
def start_live_call():
    """Start a new live AI medical consultation session"""
    try:
        cleanup_old_sessions()
        
        data = request.get_json() or {}
        
        # Manual optional JWT extraction
        user_id = None
        try:
            from flask_jwt_extended import verify_jwt_in_request, get_jwt_identity
            verify_jwt_in_request(optional=True)
            user_id = get_jwt_identity()
        except Exception:
            pass  # No valid JWT, continue without user_id
        
        specialist = data.get('specialist', 'physician')
        if specialist not in MEDICAL_SPECIALISTS:
            specialist = 'physician'
        
        session_id = str(uuid.uuid4())
        
        session = LiveCallSession(
            session_id=session_id,
            user_id=user_id,
            specialist=specialist,
            created_at=datetime.now(),
            last_activity=datetime.now(),
            language_code=data.get('language', 'en-US'),
            patient_context=data.get('patient_context', '')
        )
        
        active_sessions[session_id] = session
        
        specialist_config = MEDICAL_SPECIALISTS.get(specialist, {})
        
        return jsonify({
            "status": "success",
            "message": "Live consultation session started",
            "session_id": session_id,
            "session_info": session.to_dict(),
            "greeting": f"Hello! I'm {specialist_config.get('name', 'Dr. AI')}, your virtual {specialist_config.get('specialty', 'health')} assistant. How can I help you today?"
        })
        
    except Exception as e:
        logger.error(f"Error starting live call: {e}")
        import traceback
        traceback.print_exc()
        return jsonify({"status": "error", "message": str(e)}), 500


@live_ai_bp.route('/live-call/speech', methods=['POST'])
def process_speech():
    """Process patient speech and generate AI response with audio"""
    try:
        data = request.get_json()
        
        if not data or 'session_id' not in data or 'text' not in data:
            return jsonify({"status": "error", "message": "Missing required parameters"}), 400
        
        session_id = data['session_id']
        text = data['text'].strip()
        
        if session_id not in active_sessions:
            return jsonify({"status": "error", "message": "Invalid or expired session"}), 404
        
        session = active_sessions[session_id]
        session.last_activity = datetime.now()
        session.total_interactions += 1
        
        # Add user message to history
        session.conversation_history.append({
            'role': 'user',
            'content': text,
            'timestamp': datetime.now().isoformat()
        })
        
        # Generate AI response
        ai_response = generate_medical_response(session, text)
        
        # Add AI response to history
        session.conversation_history.append({
            'role': 'assistant',
            'content': ai_response,
            'timestamp': datetime.now().isoformat()
        })
        
        # Generate audio if TTS available
        audio_result = None
        if TTS_AVAILABLE and speech_generator:
            audio_result = speech_generator.generate_speech(
                text=ai_response,
                voice=get_optimal_voice(session.language_code),
                language_code=session.language_code
            )
        
        response = {
            "status": "success",
            "text": ai_response,
            "session_id": session_id,
            "interaction_count": session.total_interactions
        }
        
        if audio_result and audio_result.get('status') == 'success':
            # Build audio URL - use direct path without duplication
            server_url = request.url_root.rstrip('/')
            filename = audio_result.get('filename')
            response['audio_url'] = f"{server_url}/api/ai-sathi/audio/{filename}"
            response['filename'] = filename
        
        return jsonify(response)
        
    except Exception as e:
        logger.error(f"Error processing speech: {e}")
        return jsonify({"status": "error", "message": str(e)}), 500


@live_ai_bp.route('/live-call/end', methods=['POST'])
def end_live_call():
    """End the live consultation session"""
    try:
        data = request.get_json() or {}
        session_id = data.get('session_id')
        
        if not session_id:
            return jsonify({"status": "error", "message": "Missing session_id"}), 400
        
        if session_id in active_sessions:
            session = active_sessions.pop(session_id)
            duration = datetime.now() - session.created_at
            
            return jsonify({
                "status": "success",
                "message": "Consultation session ended",
                "summary": {
                    "session_id": session_id,
                    "specialist": session.specialist,
                    "duration_seconds": int(duration.total_seconds()),
                    "total_interactions": session.total_interactions
                },
                "reminder": MEDICAL_DISCLAIMER
            })
        
        return jsonify({"status": "error", "message": "Session not found"}), 404
        
    except Exception as e:
        logger.error(f"Error ending live call: {e}")
        return jsonify({"status": "error", "message": str(e)}), 500


@live_ai_bp.route('/live-call/keep-alive', methods=['POST'])
def keep_alive():
    """Keep session alive"""
    data = request.get_json() or {}
    session_id = data.get('session_id')
    
    if session_id and session_id in active_sessions:
        active_sessions[session_id].last_activity = datetime.now()
        return jsonify({"status": "success"})
    
    return jsonify({"status": "error", "message": "Session not found"}), 404


@live_ai_bp.route('/audio/<filename>')
def serve_audio(filename):
    """Serve generated audio files"""
    file_path = os.path.join(AUDIO_OUTPUT_DIR, filename)
    if os.path.exists(file_path):
        return send_file(file_path, mimetype='audio/wav')
    return jsonify({"status": "error", "message": "Audio file not found"}), 404


@live_ai_bp.route('/specialists', methods=['GET'])
def get_specialists():
    """Get available medical specialists for live consultation"""
    specialists = [
        {
            'id': key,
            'name': config['name'],
            'specialty': config['specialty'],
            'description': f"Consult with {config['name']} for {config['specialty'].lower()} concerns"
        }
        for key, config in MEDICAL_SPECIALISTS.items()
    ]
    return jsonify(specialists)
